#!/usr/bin/env python3
"""
generate_flip_counts.py

Generate a per‑link array of tick‑flip counts for a 4×4 lattice.  For each
link in the lattice a short random walk over the tick‑flip operator algebra is
performed and the number of changes at a link‑dependent context index is counted.
The resulting length‑32 array is saved to ``data/flip_counts.npy`` and printed.
"""

import os
import sys
import yaml
import numpy as np

# Ensure the ar-operator-core package is discoverable without installation.  When
# this repository lives alongside ar-operator-core (as in the integrated
# workspace) the following path manipulation adds it to the import search path.
_core_path = os.path.abspath(os.path.join(os.path.dirname(__file__), '..', '..', 'ar-operator-core'))
if _core_path not in sys.path:
    sys.path.insert(0, _core_path)

from flip_operators import F, S, X, C, Phi, TickState


def count_flips_on_link(link, sim_params):
    """Count how many flips occur on a single lattice link.

    A tiny tick chain is initialised as a delta spike at the centre of the
    distribution.  The link index is mapped to a watch index in the
    distribution so that different links probe different radii.  A fixed
    operator sequence (F, S, X, C, Φ) is applied ``steps_per_link`` times and
    every change at the watch index is counted as a flip.

    Parameters
    ----------
    link : tuple
        A tuple ``((x, y), mu)`` identifying one of the lattice links.
    sim_params : dict
        Simulation parameters containing ``N`` (context depth),
        ``steps_per_link`` and ``seed`` for the random number generator.

    Returns
    -------
    int
        The total number of flips observed on this link.
    """
    N = sim_params.get('N', 2)
    steps = sim_params.get('steps_per_link', 1000)
    seed = sim_params.get('seed', None)

    # Deterministic RNG for reproducibility
    rng = np.random.default_rng(seed)

    # Initial delta‑spike at the centre (r=0)
    dist0 = np.zeros(2 * N + 1)
    centre = N
    dist0[centre] = 1.0
    state = TickState(distribution=dist0, N=N)

    # Map link coordinates to an index in the distribution.  A simple scheme
    # distributes the 32 links across the available 2*N+1 positions.
    ((x, y), mu) = link
    watch_idx = (x + y + mu) % (2 * N + 1)

    flip_count = 0
    ops = [F, S, X, C, Phi]

    for _ in range(steps):
        for op in ops:
            new_state = op(state)
            if not np.isclose(new_state.distribution[watch_idx], state.distribution[watch_idx]):
                flip_count += 1
            state = new_state
    return flip_count


def build_default_lattice(size=4, boundary='periodic'):
    """Construct a 2D lattice with periodic boundary conditions.

    This utility mirrors the behaviour of the Volume 4 pipeline.  It returns
    a list of ``((x, y), mu)`` tuples for a ``size×size`` lattice and two
    directions ``mu ∈ {0, 1}``.
    """
    links = []
    directions = [(1, 0), (0, 1)]
    for x in range(size):
        for y in range(size):
            for mu, (dx, dy) in enumerate(directions):
                nx = x + dx
                ny = y + dy
                if boundary == 'periodic':
                    nx %= size
                    ny %= size
                else:
                    if not (0 <= nx < size and 0 <= ny < size):
                        continue
                links.append(((x, y), mu))
    return np.array(links, dtype=object)


def main():
    # Load configuration.  Resolve the config file relative to the repository
    # root so that the script can be executed from any working directory.
    repo_root = os.path.abspath(os.path.join(os.path.dirname(__file__), '..'))
    config_path = os.path.join(repo_root, 'config.yaml')
    with open(config_path) as f:
        cfg = yaml.safe_load(f)

    # Resolve data directory relative to the repository root.  Without this
    # adjustment ``data_dir`` would be interpreted relative to the current
    # working directory, which causes ``flip_counts.npy`` to be written to
    # the top‑level workspace when running from outside this repo.  If the
    # config specifies an absolute path it is used as is, otherwise it is
    # joined with the repo root.
    cfg_data_dir = cfg.get('data_dir', 'data')
    if os.path.isabs(cfg_data_dir):
        data_dir = cfg_data_dir
    else:
        data_dir = os.path.join(repo_root, cfg_data_dir)

    # lattice_file and output_file are just names under data_dir
    lattice_file = cfg.get('lattice_file', 'lattice.npy')
    out_file = cfg.get('output_file', 'flip_counts.npy')
    sim_params = cfg

    lattice_path = os.path.join(data_dir, lattice_file)

    # Build a default lattice if none exists
    if not os.path.exists(lattice_path):
        os.makedirs(data_dir, exist_ok=True)
        lattice = build_default_lattice(size=4, boundary='periodic')
        np.save(lattice_path, lattice)
    else:
        lattice = np.load(lattice_path, allow_pickle=True)

    n_links = len(lattice)
    flip_counts = np.zeros(n_links, dtype=int)

    # Compute flips for each link
    for idx, link in enumerate(lattice):
        flip_counts[idx] = count_flips_on_link(link, sim_params)

    # Save result
    out_path = os.path.join(data_dir, out_file)
    os.makedirs(data_dir, exist_ok=True)
    np.save(out_path, flip_counts)
    print(f"Wrote flip_counts.npy with shape {flip_counts.shape}, non‑zero entries: {np.count_nonzero(flip_counts)}")


if __name__ == '__main__':
    main()